;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  grid-x-inc  ;; the amount of patches in between two roads in the x direction
  grid-y-inc  ;; the amount of patches in between two roads in the y direction
  acceleration  ;; the constant that controls how much a car speeds up or slows down by if it is to accelerate or decelerate
  phase  ;; keeps track of the phase
  clock  ;; keeps track of the total times thru the go procedure
  num-cars-stopped  ;; the number of cars that are stopped during a single pass thru the go procedure
  old-display-which-metric  ;; holds the value of display-which-metric for the last time through the go procedure

  ;; patch agentsets
  intersections  ;; agentset containing the patches that are intersections
  roads  ;; agentset containing the patches that are roads

  ;; string and list variables that hold data passed to or from calculators
  wait-data  ;; list that holds the average wait time of the cars for each pass through the go procedure
  stopped-data  ;; list that holds the number of stopped of the cars for each pass through the go procedure
  speed-data  ;; list that holds the average speed of the cars for each pass through the go procedure
  time-data  ;; list that holds the value of clock for each pass through the go procedure
  ;; the following list holds sequences of two numbers that inform the calcs which stoplight they control
  ;; the two numbers, in order, are as follows: the row of the stoplight, the column of the stoplight
  locs
  ;; the following string holds the ids of the calcs that are processed.
  ;; this is used in determining which pair of numbers the calculators should use to get the location of the stoplight they control.
  ids

  ;;quick start instructions variables
  quick-start  ;; the current quickstart instruction displayed in the quickstart monitor
  qs-item  ;; the index of the current quickstart instruction
  qs-items  ;; the list of quickstart instructions
]

turtles-own
[
  speed  ;; the speed of the turtle
  up-car?  ;; this will be true if the turtle moves downwards and false if it moves to the right
  wait-time  ;; the amount of time since the last time a turtle has moved
]

patches-own
[
  intersection?  ;; this is true if the patch is at the intersection of two roads
  accident?  ;; this is true if a crash has occurred at this intersection.  this will never be true for a non-intersection patch
  green-light-up?  ;; this is true if the green light is above the intersection.  otherwise, it is false.  this is only true for patches that are intersections.
  my-row  ;; this holds the row of the intersection counting from the upper left corner of the graphics window.  it is -1 for patches that are not intersections.
  my-column  ;; this holds the column of the intersection counting from the upper left corner of the graphics window.  it is -1 for patches that are not intersections.
  user-id  ;; this holds the user-id that corresponds to the intersection.  it is -1 for patches that are not intersections.
  my-phase  ;; this holds the phase for the intersection.  it is -1 for patches that are not intersections.
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  setup true
  setup-quick-start
  hubnet-set-client-interface "TI-83+" [ "AAA - Gridlock 1.2" [ "L1" ] ]
  hubnet-reset
end

;; Initialize the display by giving the global and patch variables intial values.
;; Create num-cars of turtles if there are enough road patches for one turtle to be created
;; per road patch.
;; Setup the plots
;; All code in setup is done if full-setup? is true.  If it is false, then it doesn't clear the information
;; about the users; users still retain the same light that they had before.
to setup [ full-setup? ]
  clear-output
  if full-setup?  ;; We only want to clear the patches if the we are doing a full-setup
  [ cp ]
  ct
  clear-all-plots

  setup-globals full-setup?

  ;; First we ask the patches to draw themselves and set up a few variables
  setup-patches full-setup?

  set-default-shape turtles "car"

  if (num-cars > count roads)
  [
    user-message "There are too many cars for the amount of " +
    "road.  Either increase the amount of roads " +
    "by increasing the GRID-SIZE-X or " +
    "GRID-SIZE-Y sliders, or decrease the " +
    "number of cars by lowering the NUMBER slider.\n" +
    "The setup has stopped."
    stop
  ]

  ;; Now create the turtles and have each created turtle call the functions setup-cars and set-car-color
  cct num-cars
  [
    setup-cars
    set-car-color
    record-data
  ]

  ;; give the turtles an initial speed
  ask turtles
  [ set-car-speed ]

  update-list-info

  setup-plots
end

;; Initialize the global variables to appropriate values
to setup-globals [ full-setup? ]
  set phase 0
  set clock 0
  set num-cars-stopped 0
  if full-setup?
  [
    set grid-x-inc screen-size-x / grid-size-x
    set grid-y-inc screen-size-y / grid-size-y
  ]

  ;; initialize the lists and string for HubNet
  set wait-data []
  set stopped-data []
  set speed-data []
  set time-data []
  if full-setup?
  [
    set locs []
    set ids ""
  ]

  ;; don't make acceleration 0.1 since we could get a rounding error and end up on a patch boundary
  set acceleration 0.099
end

;; Make the patches have appropriate colors, setup the roads and intersections agentsets,
;; and initialize the traffic lights to one setting
to setup-patches [ full-setup? ]
  if full-setup?
  [
    ;; initialize the patch-own variables and color the patches to a base-color
    ask patches
    [
      set intersection? false
      set accident? false
      set green-light-up? true
      set my-row -1
      set my-column -1
      set user-id -1
      set my-phase -1

      set pcolor brown + 3
    ]

    ;; initialize the global variables that hold patch agentsets
    set roads patches with [ ( floor( (pxcor + screen-edge-x - floor(grid-x-inc - 1)) mod grid-x-inc ) = 0 ) or
                              ( floor( (pycor + screen-edge-y) mod grid-y-inc ) = 0 ) ]
    set intersections roads with [ ( floor( (pxcor + screen-edge-x - floor(grid-x-inc - 1)) mod grid-x-inc ) = 0 ) and
                                    ( floor( (pycor + screen-edge-y) mod grid-y-inc ) = 0 ) ]

    ask roads
    [ set pcolor white ]
  ]
  setup-intersections full-setup?
end

;; Give the intersections appropriate values for the intersection?, my-row, and my-column 
;; patch variables.  Make all the traffic lights start off so that the lights are red 
;; horizontally and green vertically.  
to setup-intersections [ full-setup? ]
  ask intersections
  [
    set intersection? true
    set green-light-up? true
    set my-phase 0
    if full-setup?
    [
      set my-row floor( (pycor + screen-edge-y) / grid-y-inc )
      set my-column floor( (pxcor + screen-edge-x) / grid-x-inc )
    ]
    set-signal-colors
  ]
end

;; Initialize the turtle variables to appropriate values and place the turtle on an empty road patch.
to setup-cars  ;; turtle procedure
  set speed 0
  set wait-time 0

  put-on-empty-road

  ifelse intersection?
  [
    ifelse random 2 = 1
    [ set up-car? true ]
    [ set up-car? false ]
  ]
  [
    ifelse ( floor( (pxcor + screen-edge-x - floor(grid-x-inc - 1)) mod grid-x-inc ) = 0 )
    [ set up-car? true ]
    [ set up-car? false ]
  ]

  ifelse up-car?
  [ set heading 180 ]
  [ set heading 90 ]
end

;; Find a road patch without any turtles on it and place the turtle there.
to put-on-empty-road  ;; turtle procedure
  locals
  [ road ]

  set road random-one-of roads
  setxy (pxcor-of road) (pycor-of road)
  if any? other-turtles-here
  [ put-on-empty-road ]
end

;; Initialize the plots
to setup-plots
set-current-plot "Stopped Cars"
  set-plot-y-range 0 num-cars
  plot num-cars-stopped
  
  set-current-plot "Average Wait Time of Cars"
  plot mean values-from turtles [ wait-time ]
  
  set-current-plot "Average Speed of Cars"
  set-plot-y-range 0 speed-limit
  plot mean values-from turtles [ speed ]
end

;; give the user some information about what the setup button does so they can 
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? ("The SETUP button should only be used when starting "
              + "over with a new group (such as a new set of students) since "
              + "all data is lost.  Use the RE-RUN button for continuing with "
              + "an existing group."
              + "\n\nDo you really want to setup the model?")
  [
    user-message "Before closing this dialog, please do the following:"
                 + "\n  -Have everyone that is currently logged in, log off and "
                 + "clear the calulator memory. (Press 2nd MEM 7 1 2)"
                 + "\n  -Open the teacher console for this activity and press the ERASE ALL DATA button."
    setup true
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

;; receives information from the calculators and runs the simulation
to go
  every delay
  [
    ;; get commands and data from the calculators
    listen-calc

    ;; clear any accidents from the last time thru the go procedure
    clear-accidents
    
    ;; if there are any intersections that are to switch automatically,
    ;; have them change their color
    set-signals
    set num-cars-stopped 0

    ;; set the turtles speed for this time thru the procedure, move them forward their speed,
    ;; record data for plotting and sending to the calculators, and set the color of the turtles
    ;; to an appropriate color based on their speed
    ask turtles
    [
      set-car-speed
      fd speed
      record-data
      set-car-color
    ]
    ;; crash the cars if crash? is true
    if crash?
    [ crash-cars ]

    ;; update the information in the lists sent to the calculators
    update-list-info

    ;; update the plots with the new information from this pass thru the procedure
    do-plotting
    ;; update the clock and the phase
    clock-tick
  ]
end

;; reports the amount of seconds by which to slow the model down
to-report delay
  ifelse simulation-speed <= 0
  [ report ln (10 / 0.001) ]
  [ report ln (10 / simulation-speed) ]
end

;; have the traffic lights change color if phase equals each intersections' my-phase
to set-signals
  locals [ intersections-to-change ]
  ifelse auto?
  [
    set intersections-to-change intersections with
      [phase = floor ((my-phase * ticks-per-cycle) / 100)]
  ]
  [
    set intersections-to-change intersections with
      [user-id = -1 and phase = floor ((my-phase * ticks-per-cycle) / 100)]
  ]
  ask intersections-to-change
  [
    set green-light-up? (not green-light-up?)
    set-signal-colors
  ]
end

;; This procedure checks the variable green-light-up? at each intersection and sets the
;; traffic lights to have the green light up or the green light to the left.
to set-signal-colors  ;; intersection (patch) procedure
  ifelse power?
  [
    ifelse green-light-up?
    [
      set (pcolor-of patch-at -1 0) red
      set (pcolor-of patch-at 0 1) green
    ]
    [
      set (pcolor-of patch-at -1 0) green
      set (pcolor-of patch-at 0 1) red
    ]
  ]
  [
    set (pcolor-of patch-at -1 0) white
    set (pcolor-of patch-at 0 1) white
  ]
end

;; set any intersection's color that had an accident back to white and make accident? false
to clear-accidents
  if crash?
  [
    ask patches with [ accident? ]
    [
      set pcolor white
      set accident? false
    ]
  ]
end

;; set the turtles' speed based on whether they are at a red traffic light or the speed of the
;; turtle (if any) on the patch in front of them
to set-car-speed  ;; turtle procedure
  ifelse pcolor = red
  [ set speed 0 ]
  [
    ifelse up-car?
    [ set-speed 0 -1 ]
    [ set-speed 1 0 ]
  ]
end

;; set the speed variable of the turtle to an appropriate value (not exceeding the
;; speed limit) based on whether there are turtles on the patch in front of the turtle
to set-speed [ delta-x delta-y ]  ;; turtle procedure
  locals
  [
    up-cars?-ahead
    turtles-ahead
  ]

  ;; get the turtles on the patch in front of the turtle
  ask patch-at delta-x delta-y
  [ set turtles-ahead turtles-here ]

  ;; if there are turtles in front of the turtle, slow down
  ;; otherwise, speed up
  ifelse any? turtles-ahead
  [
    set up-cars?-ahead values-from turtles-ahead [ up-car? ]
    ifelse member? up-car? up-cars?-ahead and member? (not up-car?) up-cars?-ahead
    [
      if not crash?
      [ set speed 0 ]
    ]
    [
      set speed speed-of one-of turtles-ahead
      slow-down
    ]
  ]
  [ speed-up ]
end

;; decrease the speed of the turtle
to slow-down  ;; turtle procedure
  ifelse speed <= 0  ;;if speed < 0
  [ set speed 0 ]
  [ set speed speed - acceleration ]
end

;; increase the speed of the turtle
to speed-up  ;; turtle procedure
  ifelse speed > speed-limit
  [ set speed speed-limit ]
  [ set speed speed + acceleration ]
end

;; set the color of the turtle to a different color based on how fast the turtle is moving
to set-car-color  ;; turtle procedure
  ifelse speed < ( speed-limit / 2 )
  [ set color blue ]
  [ set color cyan - 2 ]
end

;; keep track of the number of stopped turtles and the amount of time a turtle has been stopped 
;; if its speed is 0
to record-data  ;; turtle procedure
  ifelse speed = 0
  [
    set num-cars-stopped num-cars-stopped + 1
    set wait-time wait-time + 1
  ]
  [ set wait-time 0 ]
end

;; crash any turtles at the same intersection going in different directions
to crash-cars
  ask intersections with [ any? turtles-here with [ up-car? ] and any? turtles-here with [ not up-car? ] ]
  [
    set accident? true
    set pcolor orange
  ]
end

;; add the new information from this pass thru the go procedure to the HubNet lists
to update-list-info
  set wait-data lput ( mean values-from turtles [ wait-time ] ) wait-data
  set stopped-data lput num-cars-stopped stopped-data
  set speed-data lput ( mean values-from turtles [ speed ] ) speed-data
  set time-data lput clock time-data
end

;; plot the data from this pass thru the go procedure
to do-plotting
  ifelse display-which-metric = old-display-which-metric
  [
    ;; we only need to plot 1 value since the current plot is the same as the plot we are supposed to plot to now
    ifelse display-which-metric = 1
    [ plot-new-value "Stopped Cars" num-cars-stopped ]
    [
      ifelse display-which-metric = 2
      [ plot-new-value "Average Speed of Cars" mean values-from turtles [ speed ] ]
      [
        ifelse display-which-metric = 3
        [ plot-new-value "Average Wait Time of Cars" mean values-from turtles [ wait-time ] ]
        [
          plot-new-value "Stopped Cars" num-cars-stopped
          plot-new-value "Average Wait Time of Cars" mean values-from turtles [ wait-time ]
          plot-new-value "Average Speed of Cars" mean values-from turtles [ speed ]
        ]
      ]
    ]
  ]
  [
    ;; otherwise, we need to plot at least 1 list since the plot we are supposed to plot to is different from the plot we last plotted in
    ifelse display-which-metric = 1
    [ clear-plots-and-plot-in-new-plot "Stopped Cars" stopped-data ]
    [
      ifelse display-which-metric = 2
      [ clear-plots-and-plot-in-new-plot "Average Speed of Cars" speed-data ]
      [
        ifelse display-which-metric = 3
        [ clear-plots-and-plot-in-new-plot "Average Wait Time of Cars" wait-data ]
        [
          ifelse old-display-which-metric = 1
          [ plot-value-and-lists "Stopped Cars" num-cars-stopped "Average Speed of Cars" speed-data "Average Wait Time of Cars" wait-data ]
          [
            ifelse old-display-which-metric = 2
            [ plot-value-and-lists "Average Speed of Cars" (mean values-from turtles [ speed ]) "Stopped Cars" stopped-data "Average Wait Time of Cars" wait-data ]
            [ plot-value-and-lists "Average Wait Time of Cars" (mean values-from turtles [ wait-time ]) "Stopped Cars" stopped-data "Average Speed of Cars" speed-data ]
          ]
        ]
      ]
    ]
    set old-display-which-metric display-which-metric
  ]
end

to plot-new-value [ name-of-plot value ]
  set-current-plot name-of-plot
  plot value
end

to clear-plots-and-plot-in-new-plot [ name-of-plot list-to-plot ]
  clear-all-plots
  plot-new-list name-of-plot list-to-plot
end

to plot-new-list [ name-of-plot list-to-plot ]
  locals
  [ index ]
  set index 0

  set-current-plot name-of-plot
  clear-plot
  repeat length list-to-plot
  [
    plot item index list-to-plot
    set index index + 1
  ]
end

to plot-value-and-lists [ value-plot value list-plot1 list-to-plot1 list-plot2 list-to-plot2 ]
  plot-new-value value-plot value
  plot-new-list list-plot1 list-to-plot1
  plot-new-list list-plot2 list-to-plot2
end

;; increases the clock by 1 and cycles phase to the next appropriate value
to clock-tick
  set clock clock + 1
  ;; The phase cycles from 0 to ticks-per-cycle, then starts over.
  set phase phase + 1
  if phase mod ticks-per-cycle = 0
  [ set phase 0 ]
end

;; this allows the user to pick an intersection using the mouse and
;; dis-associate that intersection from a client. this is useful if a calc
;; drops a connection.
to abandon-intersection
  locals
  [
    x-mouse  ;; the x coordinate of the mouse's position when it is down
    y-mouse  ;; the y coordinate of the mouse's position when it is down
  ]
  if( mouse-down? )
  [
    set x-mouse ( round mouse-xcor )
    set y-mouse ( round mouse-ycor )
  ]
  if( intersection?-of patch-at x-mouse y-mouse )
  [
    ask intersections with [ pxcor = x-mouse and pycor = y-mouse ]
    [
      set user-id -1
      set my-phase 0
      set plabel-of patch-at -1 1 no-label
    ]
    stop
  ]
end

;; update the plots to show the current value of display-which-metric
to show-current-metric-in-plots
  if display-which-metric != old-display-which-metric
  [
    ;; otherwise, we need to plot at least 1 list since the plot we are supposed to plot to is different from the plot we last plotted in
    ifelse display-which-metric = 1
    [ clear-plots-and-plot-in-new-plot "Stopped Cars" stopped-data ]
    [
      ifelse display-which-metric = 2
      [ clear-plots-and-plot-in-new-plot "Average Speed of Cars" speed-data ]
      [
        ifelse display-which-metric = 3
        [ clear-plots-and-plot-in-new-plot "Average Wait Time of Cars" wait-data ]
        [
          ifelse old-display-which-metric = 1
          [
            plot-new-list "Average Speed of Cars" speed-data
            plot-new-list "Average Wait Time of Cars" wait-data
          ]
          [
            ifelse old-display-which-metric = 2
            [
              plot-new-list "Stopped Cars" stopped-data
              plot-new-list "Average Wait Time of Cars" wait-data
            ]
            [
              plot-new-list "Stopped Cars" stopped-data
              plot-new-list "Average Speed of Cars" speed-data
            ]
          ]
        ]
      ]
    ]
    set old-display-which-metric display-which-metric
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, calculators, and TISchool webpage to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to setup the HubNet activity."
    "Optional- Zoom In (see Tools in the Menu Bar) - optional"
    "Change the traffic grid (using the sliders GRID-SIZE-X and..." "GRID-SIZE-Y) to make enough lights for everyone."
    "Change any other of the settings that you would like to change."
    "For example, if you plan on running Gridlock in..." "the MANUAL mode, be sure to have AUTO? set to OFF."
    "Also be sure to set CALCULATOR-DATA-SET and..." "METRIC-PASSED to the appropriate values."
    "Press the SETUP NetLogo button."
    "Press the INITIAL LOGIN button."
    "Everyone: Reset the RAM on your calculator (2nd MEM 7 1 2)."
    "Login to the calculator."
    "Teacher: Once everyone is at the calculator Main Menu..." "stop the INITIAL LOGIN button by pressing it again."
    "Everyone: Whichever mode AUTO? is set for in NetLogo,..." "choose that option on the calculators at the calculator Main Menu."
    "If you have chosen MANUAL,..."
    "you can change the state of your light by pressing..." "ENTER or the CHNG softkey on the calculator."
    "If you have chosen AUTO,..."
    "you can change the phase of your light by pressing..." "a number in the range 00-99."
    "Teacher: Once everyone is in the appropriate section,..." "start the simulation by pressing the GO button."
    "Everyone: After a while of running the simulation,..." "you may want to view the data that NetLogo could send."
    "Press the BACK softkey to return to the Main Menu."
    "Teacher: Once everyone is back in the calculator Main Menu..." "press the NetLogo GO button to stop it."
    "Make sure that CALCULATOR-DATA-SET and METRIC-PASSED..." "are set to the appropriate values that you want to..." "send to the calculators."
    "Press the SEND DATA button."
    "Everyone: To view the data sent to the calculators,..." "choose the VIEW DATA option."
    "Teacher: To send different data to the calculators,..." "change the values of CALCULATOR-DATA-SET and..." "METRIC-PASSED and press the SEND DATA button."
    "Everyone: To collect more data, choose the..." "calculator menu option that corresponds to..." "the AUTO? switch and continue."
    "Teacher: Start the model by pressing the NetLogo GO button."
    "To change the mode of AUTO?,..."
    "Everyone: Exit to the calculator Main Menu."
    "Teacher: Change the value of AUTO?."
    "Press the NetLogo RE-RUN button."
    "Everyone: Whichever mode AUTO? is now set for in NetLogo,..." "choose that option on the calculators at the calculator Main Menu..." "and continue."
    "To change the metric plotted in NetLogo,..."
    "Select the CHOOSE METRIC option at the calculator Main Menu."
    "Press 1 to see the STOPPED CARS plot, 2 for..." "the AVERAGE SPEED plot, 3 for the AVERAGE WAIT plot, and..." "4 for all three plots."

    "Teacher: To rerun the activity with the same group,..." "do not clear the server."
    "Everyone: Exit to the calculator Main Menu."
    "Teacher: In NetLogo, press the NetLogo GO button, if it is on."
    "Change the values of the sliders and switches to..." "the values you want."
    "Press the RE-RUN button."
    "Everyone: Choose the appropriate option from..." "the calculator Main Menu based on the value of AUTO?."
    "Teacher: Once everyone is in the appropriate section,..." "restart the simulation by pressing the GO button."
    
    "Teacher: To start the simulation over with a new group,..." "stop the model by pressing the NetLogo GO button, if it is on."
    "Have everyone, including yourself, logout of their calculators."
    "Press the ERASE ALL DATA button on the TISCHOOL site."
    "Press the NetLogo SETUP button."
    "Follow these instructions from the beginning again."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the calculators ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; when a command is sent, find out which calculator sent it and then execute the command
to listen-calc
  locals
  [
    cmd
    current-id
  ]

  set cmd -1

  ;; use without-interruption to eliminate the possiblity that if both the GO and 
  ;; INITIAL-LOGIN buttons are pressed that they both try processing the same message
  without-interruption
  [
    while [ hubnet-message-waiting? ]
    [
      hubnet-fetch-message
      set current-id hubnet-message-source
      set cmd item 0 hubnet-message
      ifelse cmd = 0
      [
        give-intersection-coords current-id
        wait 1  ;; we want to give some time for other calcs to log in on this round
      ]
      [
        ifelse cmd = 1
        [ calc-manual current-id ]
        [
          ifelse cmd = 2
          [ calc-auto current-id ]
          [ switch-metric-view ]
        ]
      ]
    ]

    ;; if the command was to log in calcs, then we want to send the updated string of id's and the list of locations in the activity
    if cmd = 0
    [
      hubnet-broadcast "STR8" ids
      hubnet-broadcast "LOCS" locs
    ]
  ]
end

;; when a calculator is logged into, if there are free intersections, 
;; assign a free intersection to that calculator.
;; if this current-id already has an intersection, give the calculator the same intersection.
to give-intersection-coords [ current-id ]
  ifelse( member? ( "," + current-id + "," ) ( "," + ids + "," ) )
  [
    if not any? intersections with [ user-id = current-id ]
    [
      ;; the case where they tried logging in previously but there was no room for them
      remove-id-and-location current-id
      get-free-intersection current-id
    ]
    ;; for the case where they logged in previously but at some point got kicked out 
    ;; (i.e. there is an intersection with user-id = current-id) and are now trying to log back in, 
    ;; we don't need to do anything since the user-id is already in the list ids and the 
    ;; coordinates for the intersection are already in the list locs
  ]
  [
    ;; they haven't logged in before
    get-free-intersection current-id
  ]
end

;; remove current-id from ids and remove the location values in locs corresponding to 
;; this current-id
to remove-id-and-location [ current-id ]
  locals
  [
    index  ;; the number of commas found in ids before the position of current-id
    i  ;; current place we are looking in ids
  ]
  set index 0
  set i 0
  if( not member? ( "," + current-id + "," ) ( "," + ids + "," ) )
  [ stop ]

  ;; get the number of the commas before current-ids in ids
  repeat position ( "," + current-id + "," ) ( "," + ids + "," )
  [
    if( "," = item i ids )
    [
      set index index + 1
    ]
    set i i + 1
  ]

  ;; remove current-id from ids
  ifelse member? ( current-id + "," ) ids
  [ set ids remove ( current-id + "," ) ids ]
  [
    ifelse member? ( "," + current-id ) ids
    [ set ids remove ( "," + current-id ) ids ]
    [ set ids remove current-id ids ]
  ]

  ;; remove the location values from locs
  set locs replace-item ( 2 * index ) locs "remove this string"
  set locs replace-item ( 2 * index + 1 ) locs "remove this string"
  set locs remove "remove this string" locs
end

;; if there are any free intersections, pick one of them at random and give it to the 
;; current-id.
;; if there are not any free intersections, toss an error and put error values into the list
to get-free-intersection [ current-id ]
  ifelse any? intersections with [ user-id = -1 ]
  [
    ;; pick a random intersection that hasn't been taken yet
    ask random-one-of intersections with [ user-id = -1 ]
    [
      set user-id current-id
      add-id-and-location current-id my-column my-row
      ask patch-at -1 1
      [
        set plabel-color black
        set plabel current-id
      ]
    ]
  ]
  [
    user-message "Not enough lights for student with id: " + current-id
    add-id-and-location current-id -1 -1
  ]
end

;; add current-id to the end of ids and put xloc and yloc at the end of locs
to add-id-and-location [ current-id xloc yloc ]
  ifelse ids = ""
  [ set ids current-id ]
  [ set ids ids + "," + current-id ]
  set locs lput xloc locs
  set locs lput yloc locs
end

;; switch the traffic lights at the intersection for the calculator 
;; with user-id
to calc-manual [ current-id ]
  if not auto?
  [
    ask intersections with [ user-id = current-id ]
    [
      set green-light-up? (not green-light-up?)
      set-signal-colors
    ]
  ]
end

;; change the value of the phase for the intersection with user-id equalling current-id to 
;; the value passed by the calculator
to calc-auto [ current-id ]
  ask intersections with [ user-id = current-id ]
  [
    ;; we get the second item in the list since it contains the new phase of the light
    set my-phase item 1 hubnet-message
  ]
end

to switch-metric-view
  locals
  [ metric-value-from-calc ]
  set metric-value-from-calc item 1 hubnet-message

  if display-which-metric >= 1 and display-which-metric <= 3 and member? metric-value-from-calc [ 1 2 3 4 ]
  [ set display-which-metric metric-value-from-calc ]
end

;; send the data from a plot to the calculators
;; the plot is chosen by the metric-passed slider and the data set 
;; the information from the plot is stored in is chosen by the 
;; calculator-data-set slider.
;; note: that each list sent to the calculators is sampled to reduce its size to 
;; be able to be usable by the calculators.
to send-calc-data
  ifelse metric-passed = 1
  [
    hubnet-broadcast ("DATA" + calculator-data-set) sample-list stopped-data false
    hubnet-broadcast ("TIME" + calculator-data-set) sample-list time-data true
  ]
  [
    ifelse metric-passed = 2
    [
      hubnet-broadcast ("DATA" + calculator-data-set) sample-list speed-data false
      hubnet-broadcast ("TIME" + calculator-data-set) sample-list time-data true
    ]
    [
      hubnet-broadcast ("DATA" + calculator-data-set) sample-list wait-data false
      hubnet-broadcast ("TIME" + calculator-data-set) sample-list time-data true
    ]
  ]
end

;; this reports a list that has elements removed from it in a periodic manner, if the list is over a maximum length.
;; this sampling is necessary due to the limitations on the lengths of lists in the calculators.
;; this is also necessary so that the time that it takes to send the data over the network is not prohibitively long.
to-report sample-list [ list-to-sample sample-sequentially? ]
  locals
  [
    max-length  ;; this holds the maximum length that the reported list can be
    list-to-send  ;; this holds the sampled list.  this list is reported.
    index
    increment  ;; the value we increase index by each time through the loop to get the proper element of the list
  ]

  set max-length 99  ;; the TI-83+ calculators have a maximum list length of 999.  make sure that if you change this value, the new value is <= 999.
  set index 0
  set list-to-send []

  ;; if the list to be sampled is short enough, report that list
  if length list-to-sample <= max-length
  [ report list-to-sample ]

  ifelse sample-sequentially?
  [ set increment 1 ]
  [ set increment ( ( length list-to-sample ) / max-length ) ]

  repeat max-length
  [
    set list-to-send lput ( item ( floor index ) list-to-sample ) list-to-send
    set index index + increment
  ]

  report list-to-send
end


; ***NetLogo Model Copyright Notice***

; This activity and associated models and materials was created as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
; The project gratefully acknowledges the support of the 
; National Science Foundation ( REPP program ) -- grant number REC #9814682.

; Copyright 1999 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution for
; profit.

; To refer to this model in academic publications, please use:
; Wilensky, U. & Stroup, W. (1999).  NetLogo HubNet Gridlock model.
; http://ccl.northwestern.edu/netlogo/models/HubNetGridlock.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.

; In other publications, please use:
; Copyright 1998 by Uri Wilensky and Walter Stroup.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/HubNetGridlock
; for terms of use.
;
; ***End NetLogo Model Copyright Notice***
@#$#@#$#@
GRAPHICS-WINDOW
296
97
676
498
18
18
10.0
1
12
1
1
1
0

CC-WINDOW
5
703
888
798
Command Center

TEXTBOX
4
420
292
464
This slider determines which plot is drawn.  1=stopped cars, 2=average speed, 3=average wait, 4=all three plots

SLIDER
3
464
152
497
display-which-metric
display-which-metric
1
4
1
1
1
NIL

BUTTON
109
243
204
276
Initial Login
listen-calc
T
1
T
OBSERVER
T
NIL

BUTTON
111
286
286
319
Revert light to unmanned
abandon-intersection
T
1
T
OBSERVER
T
NIL

BUTTON
10
286
91
319
Send Data
send-calc-data
NIL
1
T
OBSERVER
T
NIL

TEXTBOX
145
366
296
412
Metric sent to calculators: 1=stopped cars, 2=average speed, 3=average wait

SLIDER
149
332
277
365
metric-passed
metric-passed
1
3
1
1
1
NIL

PLOT
588
500
879
689
Average Wait Time of Cars
Time
Average Wait
0.0
100.0
0.0
5.0
true
false
PENS
"default" 1.0 0 -65536 true

PLOT
295
500
586
689
Average Speed of Cars
Time
Average Speed
0.0
100.0
0.0
1.0
true
false
PENS
"default" 1.0 0 -65536 true

TEXTBOX
1
367
125
413
This controls to which plot in the calculators data is passed

SLIDER
1
332
147
365
calculator-data-set
calculator-data-set
1
3
1
1
1
NIL

BUTTON
299
60
417
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

BUTTON
590
60
674
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
513
60
591
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
T
NIL

MONITOR
286
10
685
59
Quick Start Instructions (more details in info tab)
quick-start
0
1

SLIDER
97
41
194
74
grid-size-y
grid-size-y
1
9
5
1
1
NIL

SLIDER
1
41
95
74
grid-size-x
grid-size-x
1
9
5
1
1
NIL

SWITCH
104
113
194
146
auto?
auto?
1
1
-1000

SWITCH
201
113
291
146
crash?
crash?
1
1
-1000

SWITCH
1
113
96
146
power?
power?
0
1
-1000

SLIDER
1
77
282
110
num-cars
num-cars
0
400
200
1
1
NIL

PLOT
2
500
293
689
Stopped Cars
Time
Stopped Cars
0.0
100.0
0.0
100.0
true
false
PENS
"default" 1.0 0 -65536 true

BUTTON
221
243
285
276
Go
go
T
1
T
OBSERVER
T
NIL

SLIDER
1
149
153
182
simulation-speed
simulation-speed
0
10.0
7.5
0.1
1
NIL

BUTTON
197
41
281
74
Setup
setup-prompt
NIL
1
T
OBSERVER
T
NIL

SLIDER
155
149
295
182
speed-limit
speed-limit
0.0
1.0
1.0
0.1
1
NIL

BUTTON
10
243
91
276
Re-Run
setup false
NIL
1
T
OBSERVER
T
NIL

MONITOR
187
187
292
236
Current Phase
phase
3
1

SLIDER
6
198
178
231
ticks-per-cycle
ticks-per-cycle
1
100
20
1
1
NIL

BUTTON
154
465
293
498
Show Current Plot
show-current-metric-in-plots
NIL
1
T
OBSERVER
T
NIL

@#$#@#$#@
WHAT IS IT?
-----------
Students control traffic lights in a real-time traffic simulation.  The teacher controls overall variables, such as the speed limit and the number of cars.  This allows students to explore traffic dynamics, which can lead into many areas of study, from calculus to social studies.

Challenge the students to develop strategies to improve traffic and discuss the different ways to measure the quality of traffic.

The coordinates for the traffic lights are based on the first quadrant of the Cartesian plane.  Therefore, the traffic light with the coordinates (0,0) is in the lowest row and the left-most column.  The traffic light above it has coordinates (0,1) and the traffic light to the right of it has (1,0).

This activity requires the TI Navigator activity AAA - Gridlock 1.2

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------

QUICKSTART INSTRUCTIONS:
------------------------
Contains instructions as to how to quickly setup the model, calculators, and TISchool web page so as to run this activity.  The instructions can be found below:

Teacher: Open the TISchool site in your web browser.
Enable activity AAA - Gridlock 1.2
Open the teacher console and press the ERASE ALL DATA button.

If it is not open already, open the NetLogo model.
If you are prompted by a Login dialog, enter your teacher id and password and any other necessary information.
Optional- Zoom In (see Tools in the Menu Bar)
Change the traffic grid (using the sliders GRID-SIZE-X and GRID-SIZE-Y) to make enough lights for everyone.
Change any other of the settings that you would like to change.
For example, if you plan on running Gridlock in the MANUAL mode, be sure to have AUTO? set to OFF.
Also be sure to set CALCULATOR-DATA-SET and. METRIC-PASSED to the appropriate values.
Press the SETUP NetLogo button.
Press the NetLogo INITIAL LOGIN button.

Everyone: Reset the RAM on the calculator by pressing the following keys: 2nd MEM 7 1 2.
Login to the calculator.

Teacher: Once everyone is at the calculator Main Menu, stop the INITIAL LOGIN button by pressing it again.
If there are extra intersections that are not associated with a calculator, they will switch automatically at the beginning of each cycle.

Everyone: Whichever mode AUTO? is set for in NetLogo, choose that option on the calculators at the calculator Main Menu.
If you have chosen MANUAL, you can change the state of your light by pressing ENTER or the CHNG softkey on the calculator.
If you have chosen AUTO, you can change the phase of your light by pressing a number in the range 00-99.

Teacher: Once everyone is in the appropriate section, start the simulation by pressing the GO button.

Everyone: After a while of running the simulation, you may want to view the data that NetLogo could send.
Press the BACK softkey to return to the Main Menu.

Teacher: Once everyone is back in the calculator Main Menu press the NetLogo GO button to stop it.
Make sure that CALCULATOR-DATA-SET and METRIC-PASSED are set to the appropriate values that you want to send to the calculators.
Press the SEND DATA button.

Everyone: To view the data sent to the calculators, choose the VIEW DATA option.
Teacher: To send different data to the calculators, change the values of CALCULATOR-DATA-SET and METRIC-PASSED and press the SEND DATA button.
Everyone: To collect more data, choose the calculator menu option that corresponds to the AUTO? switch and continue.
Teacher: Start the model by pressing the NetLogo GO button.

To change the mode of AUTO?,
Everyone: Exit to the calculator Main Menu
Teacher: Change the value of AUTO?.
Everyone: Whichever mode AUTO? is now set for in NetLogo, choose that option on the calculators at the calculator Main Menu and continue.

To change the metric plotted in NetLogo,
Select the CHOOSE METRIC option at the calculator Main Menu.
Press 1 to see the STOPPED CARS plot, 2 for the AVERAGE SPEED plot, 3 for the AVERAGE WAIT plot, and 4 for all three plots.

To rerun the activity with the same group,
Teacher: Do not clear the server.
Everyone: Exit to the calculator Main Menu.
Teacher: In NetLogo, stop the GO button if it is on by pressing it.
Change the values of the sliders and switches to the values you want.
Press the RE-RUN button.
Everyone: Choose the appropriate option from the calculator Main Menu based on the value of AUTO?.
Teacher: Once everyone is in the appropriate section, restart the simulation by pressing the GO button.

To start the simulation over with a new group,
Teacher: Stop the model by pressing the NetLogo GO button, if it is on.
Have everyone, including yourself, logout of their calculators.
Press the ERASE ALL DATA button on the TISCHOOL site.
Press the NetLogo SETUP button.
Follow these instructions from the beginning again.

BUTTONS:
--------
SETUP - generates a new traffic grid based on the current GRID-SIZE-X and GRID-SIZE-Y and NUM-CARS number of cars.  This also clears all the plots.  This should only be pressed when starting out with a new group of users since all users associated with a traffic light will be disassociated from that traffic light.
RE-RUN - allows you to keep the current traffic light/calculator associations, but clears the plots and after deleting all the currently existing cars, creates NUM-CARS number of cars.  This does not change the grid size.  This should be used to setup the model again for collecting more data or running the model again with the same users connected.
GO - runs the simulation indefinitely
SEND DATA - sends the data accrued in the list specified by METRIC-PASSED to the calculator data set specified by CALCULATOR-DATA-SET.  Further, if the list sent is larger than a maximum value, the list that is sent is "sampled."  This means that items are removed from the list that is sent in a periodic manner so until there is at most the maximum number of elements.  Sampling is necessary since the calculator can only handle a certain amount of data.
INITIAL LOGIN - allows users to log into the activity without running the model or collecting data
REVERT LIGHT TO UNMANNED - allows you to force a light to begin switching automatically.  This is handy when a calculator drops its connection and the corresponding light is holding up traffic.
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

SLIDERS:
--------
SPEED-LIMIT - sets the maximum speed for the cars
NUM-CARS - the number of cars in the simulation (you must press the SETUP or RE-RUN buttons to see the change)
SIMULATION-SPEED - the speed at which the simulation runs
TICKS-PER-CYCLE - sets the maximum value that the phase can be.  This has no effect when the model is run with AUTO? false.  Also, the phase that each user chooses is scaled to be less than or equal to this value.
GRID-SIZE-X - sets the number of vertical roads there are (you must press the SETUP button to see the change)
GRID-SIZE-Y - sets the number of horizontal roads there are (you must press the SETUP button to see the change)
CALCULATOR-DATA-SET - the number of the graph in the calculator to which the current data will be sent and displayed on the calculators when the VIEW DATA menu option is chosen
METRIC-PASSED - determines which plot data to send to the calculators.  1 = the data from STOPPED CARS, 2 = the data from AVERAGE SPEED OF CARS, 3 = the data from AVERAGE WAIT TIME OF CARS
DISPLAY-WHICH-METRIC - determines which plot is drawn in NetLogo.  1=STOPPED CARS, 2=AVERAGE SPEED OF CARS, 3=AVERAGE WAIT TIME OF CARS, 4=all three plots.  It should be noted that if this is 4, the calculators cannot change this value.

SWITCHES:
---------
CRASH? - toggles car crashing
POWER? - toggles the presence of traffic lights
AUTO? - toggles between automatic mode, where the students' lights change on a cycle, and manual in which students directly control the lights with their calculators.  Lights which are not associated with students always change on a cycle.

PLOTS:
-------------
STOPPED CARS - displays the number of stopped cars over time
AVERAGE SPEED OF CARS - displays the average speed of cars over time
AVERAGE WAIT TIME OF CARS - displays the average time cars are stopped over time


CALCULATOR INFORMATION
----------------------

TEACHER CALCULATOR:
-------------------
Students and teacher have identical calculator programs.

STUDENT CALCULATOR:
-------------------
After logging in, in the calculator Main Menu it is possible to choose between manual and automatic mode.  IMPORTANT!! The mode selected needs to match with the mode chosen in the NetLogo model via the AUTO? switch.  In MANUAL mode, press ENTER or the CHNG softkey to switch the state of the light you control.  In AUTO mode, enter the phase for your light.  The phase must be in the range 0-99.

You can also choose to view plot data passed back to the calculators by choosing the VIEW DATA option.  Be sure that data has been sent to the calculators by pressing the SEND DATA button in NetLogo first.

It is possible to pick which metric is plotted in the NetLogo plots by choosing the CHOOSE METRIC option.  Press the number 1 to show the STOPPED DATA plot, 2 to show the AVERAGE SPEED OF CARS plot, 3 to show the AVERAGE WAIT TIME OF CARS plot, or 4 to show all three plots simultaneously.  It should be noted that if this is 4, the calculators cannot change this value.


THINGS TO NOTICE
----------------
When cars have stopped at a traffic light, and then they start moving again, the traffic jam will move backwards even though the cars are moving forwards.  Why is this?  Discuss in your class possible reasons for this phenomena.


THINGS TO TRY
-------------
Try changing the speed limit for the cars.  How does this affect the overall efficiency of the traffic flow?  Are fewer cars stopping for a shorter amount of time?  Is the average speed of the cars higher or lower than before?

Try changing the number of cars on the roads.  Does this affect the efficiency of the traffic flow?

How about changing the speed of the simulation?  Does this affect the efficiency of the traffic flow?

Using HubNet, try running this simulation with AUTO? being true and AUTO? being false.  Is it harder to make the traffic move well using one scheme or the other?  Why?

Using HubNet, try running this simulation with AUTO? being true.  Try to find a way of setting the phases of the traffic lights so that the average speed of the cars is the highest.  Now try to minimize the number of stopped cars.  Now try to decrease the average wait time of the cars.  Is there any correlation between these different metrics?


EXTENDING THE MODEL
-------------------
Currently, the maximum speed limit (found in the SPEED-LIMIT slider) for the cars is 1.0.  This is due to the fact that the cars must look ahead the speed that they are traveling to see if there are cars ahead of them.  If there aren't, they speed up.  If there are, they slow down.  Looking ahead for a value greater than 1 is a little bit tricky.  Try implementing the correct behavior for speeds greater than 1.

The way that NetLogo samples the data lists sent to the calculators is a very simple method of sampling lists.  Try implementing a more complicated way of sampling the lists.  For instance, you could sample a small portion of the list.


CREDITS AND REFERENCES
----------------------
This activity and associated models and materials was created as part of the project:
PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
The project gratefully acknowledges the support of the National Science Foundation ( REPP program ) -- grant number REC #9814682.

Copyright 1999 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

Permission to use, copy, or modify this software and its associated documentation, models and curricular materials for educational and research purposes only and without fee is hereby granted, provided that this copyright notice and the original authors' names appear on all copies and supporting documentation. For any other uses of this software, in original or modified form, including, but not limited to distribution in whole or in part, specific prior permission must be obtained from Uri Wilensky and Walter Stroup. These programs shall not be used, rewritten, or adapted as the basis of a commercial or hardware product without first obtaining appropriate licenses from Uri Wilensky & Walter Stroup. We make no representations about the suitability of this software for any purpose. It is provided "as is" without express or implied warranty.

To refer to this model in academic publications, please use: Wilensky, U. & Stroup, W. (1999).  NetLogo HubNet Gridlock model. http://ccl.northwestern.edu/netlogo/models/HubNetGridlock. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 1999 by Uri Wilensky and Walter Stroup.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetGridlock for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
true
0
Polygon -7566196 true true 180 15 164 21 144 39 135 60 132 74 106 87 84 97 63 115 50 141 50 165 60 225 150 285 165 285 225 285 225 15 180 15
Circle -16777216 true false 180 30 90
Circle -16777216 true false 180 180 90
Polygon -16777216 true false 80 138 78 168 135 166 135 91 105 106 96 111 89 120
Circle -7566196 true true 195 195 58
Circle -7566196 true true 195 47 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
